/**
 * Platform Detection & Token Extraction
 * 
 * Detects which crypto trading platform the user is on
 * and extracts the current token address / trading pair.
 */

const PlatformDetector = (() => {
    const PLATFORMS = {
        DEXSCREENER: {
            name: 'DexScreener',
            hostMatch: 'dexscreener.com',
            icon: '📊',
            extract: (url) => {
                // ... existing ...
                const match = url.pathname.match(/^\/(solana|ethereum|bsc|arbitrum|polygon|base|avalanche|optimism)\/([a-zA-Z0-9]+)/i);
                if (match) {
                    return {
                        chain: match[1].toLowerCase(),
                        tokenAddress: match[2],
                        symbol: null,
                    };
                }
                return null;
            },
            getSymbolFromPage: () => {
                // ... existing ...
                const headerEl = document.querySelector('h2.chakra-heading')
                    || document.querySelector('[class*="pair-name"]')
                    || document.querySelector('h1');
                if (headerEl) {
                    const text = headerEl.textContent.trim();
                    const parts = text.split('/');
                    return parts[0].trim().split(' ')[0];
                }
                return null;
            },
            getTokenMintFromPage: () => {
                // Try multiple selectors for modern DexScreener layout
                // 1. data-address attributes and contract links
                const selectors = [
                    '[data-address]',
                    'a[href*="solscan.io/token/"]',
                    'a[href*="solscan.io/account/"]',
                    'a[href*="solana.fm/address/"]',
                    'a[href*="explorer.solana.com/address/"]',
                    '[class*="contract"] a',
                    'button[class*="copy"]',
                ];
                for (const sel of selectors) {
                    const els = document.querySelectorAll(sel);
                    for (const el of els) {
                        const addr = el.getAttribute('data-address')
                            || (el.href && el.href.match(/(?:token|account|address)\/([a-zA-Z0-9]{32,44})/)?.[1])
                            || null;
                        if (addr && addr.length >= 32 && addr.length <= 44 && /^[a-zA-Z0-9]+$/.test(addr)) {
                            return addr;
                        }
                    }
                }
                // 2. Look for any text that looks like a Solana address near token info
                const infoEls = document.querySelectorAll('[class*="token-info"], [class*="pair-info"], [class*="TokenInfo"]');
                for (const el of infoEls) {
                    const match = el.textContent.match(/([1-9A-HJ-NP-Za-km-z]{32,44})/);
                    if (match) return match[1];
                }
                return null;
            },
            getTokenUrl: (tokenAddress) => `https://dexscreener.com/solana/${tokenAddress}`,
            getPriceFromPage: () => {
                // Title format: 0.00234 SOL | Token/SOL ...
                // Or: 0.00234 USDC ...
                const match = document.title.match(/^[\$]?([0-9,]+\.?[0-9]*)/);
                if (match) return parseFloat(match[1].replace(/,/g, ''));
                return null;
            },
            navigate: (tokenAddress, symbol) => {
                const target = `https://dexscreener.com/solana/${tokenAddress}`;
                if (window.location.href !== target) {
                    window.location.href = target;
                }
            },
            resolve: async (address) => {
                // Check if address is a PAIR address by querying DexScreener API
                try {
                    const response = await fetch(`https://api.dexscreener.com/latest/dex/pairs/solana/${address}`);
                    if (response.ok) {
                        const data = await response.json();
                        if (data.pair && data.pair.baseToken && data.pair.baseToken.address) {
                            console.log('[FAFO] Resolved Pair Address', address, 'to Token Mint', data.pair.baseToken.address);
                            return data.pair.baseToken.address;
                        }
                    }
                } catch (e) {
                    console.warn('[FAFO] Failed to resolve DexScreener pair address', e);
                }
                return address;
            }
        },

        BIRDEYE: {
            name: 'Birdeye',
            hostMatch: 'birdeye.so',
            icon: '🦅',
            extract: (url) => {
                // ... existing ...
                const match = url.pathname.match(/\/token\/([a-zA-Z0-9]+)/i);
                if (match) {
                    const params = new URLSearchParams(url.search);
                    return {
                        chain: params.get('chain') || 'solana',
                        tokenAddress: match[1],
                        symbol: null,
                    };
                }
                return null;
            },
            getSymbolFromPage: () => {
                // ... existing ...
                const symbolEl = document.querySelector('[class*="token-symbol"]')
                    || document.querySelector('[class*="TokenSymbol"]')
                    || document.querySelector('h1 span');
                if (symbolEl) return symbolEl.textContent.trim();
                return null;
            },
            getTokenUrl: (tokenAddress) => `https://birdeye.so/token/${tokenAddress}?chain=solana`,
            getPriceFromPage: () => {
                // Title: $0.1234 | Token ...
                const match = document.title.match(/^\$([0-9,]+\.?[0-9]*)/);
                if (match) return parseFloat(match[1].replace(/,/g, ''));
                // Selector
                const el = document.querySelector('[class*="price"]');
                if (el) {
                    const txt = el.textContent.trim().replace('$', '').replace(/,/g, '');
                    const val = parseFloat(txt);
                    if (!isNaN(val)) return val;
                }
                return null;
            },
            getTokenUrl: (tokenAddress) => `https://birdeye.so/token/${tokenAddress}?chain=solana`,
            getPriceFromPage: () => {
                // Title: $0.1234 | Token ...
                const match = document.title.match(/^\$([0-9,]+\.?[0-9]*)/);
                if (match) return parseFloat(match[1].replace(/,/g, ''));
                // Selector
                const el = document.querySelector('[class*="price"]');
                if (el) {
                    const txt = el.textContent.trim().replace('$', '').replace(/,/g, '');
                    const val = parseFloat(txt);
                    if (!isNaN(val)) return val;
                }
                return null;
            },
            navigate: (tokenAddress, symbol) => {
                const target = `https://birdeye.so/token/${tokenAddress}?chain=solana`;
                if (window.location.href !== target) {
                    window.location.href = target;
                }
            }
        },

        JUPITER: {
            name: 'Jupiter',
            hostMatch: 'jup.ag',
            icon: '🪐',
            extract: (url) => {
                // /swap/SOL-TokenMint or /swap/USDC-TokenMint
                const swapMatch = url.pathname.match(/\/swap\/[^-]+-([a-zA-Z0-9]+)/i);
                if (swapMatch) {
                    return {
                        chain: 'solana',
                        tokenAddress: swapMatch[1],
                        symbol: null,
                    };
                }
                const tokenMatch = url.pathname.match(/\/tokens\/([a-zA-Z0-9]+)/i);
                if (tokenMatch) {
                    return {
                        chain: 'solana',
                        tokenAddress: tokenMatch[1],
                        symbol: null,
                    };
                }
                return null;
            },
            getSymbolFromPage: () => {
                // Jupiter /tokens/ page title: "RATHBUN ↓ $0.0014732 | Jupiter"
                const tokenPageMatch = document.title.match(/^([A-Za-z0-9]+)\s*[↑↓↕]/);
                if (tokenPageMatch) return tokenPageMatch[1];

                // Jupiter swap title: "Jupiter | Swap SOL to TOKEN"
                const swapMatch = document.title.match(/(?:Swap\s+\w+\s+to\s+)(\w+)/i);
                if (swapMatch) return swapMatch[1];

                // Try h1 element (token pages often have token name in h1)
                const h1 = document.querySelector('h1');
                if (h1) {
                    const text = h1.textContent.trim();
                    // h1 might be just the symbol like "RATHBUN" or "RATHBUN Token"
                    const symMatch = text.match(/^([A-Za-z0-9]+)/);
                    if (symMatch && symMatch[1].length <= 12 && symMatch[1] !== 'Jupiter') return symMatch[1];
                }

                // Try common Jupiter DOM selectors for token name
                const selectors = [
                    'button[class*="SwapTo"] span',
                    '[class*="output"] [class*="token"] span',
                    '[data-testid="output-token-name"]',
                    '[class*="outputToken"] button span',
                ];
                for (const sel of selectors) {
                    const el = document.querySelector(sel);
                    if (el) {
                        const text = el.textContent.trim();
                        if (text && text.length <= 12 && text !== 'SOL' && text !== 'USDC') return text;
                    }
                }

                // Fallback: check meta tags
                const metaDesc = document.querySelector('meta[property="og:title"]');
                if (metaDesc) {
                    const match = metaDesc.content.match(/(?:to|buy)\s+(\w+)/i);
                    if (match) return match[1];
                }

                return null;
            },
            getTokenUrl: (tokenAddress) => `https://jup.ag/tokens/${tokenAddress}`,
            getPriceFromPage: () => {
                // Jupiter /tokens/ page: price in title "RATHBUN ↓ $0.0014732 | Jupiter"
                const titleMatch = document.title.match(/\$([0-9,.]+)/);
                if (titleMatch) {
                    const price = parseFloat(titleMatch[1].replace(/,/g, ''));
                    if (!isNaN(price) && price > 0) return price;
                }

                // Jupiter DOM: price in ReadableNumber span
                const priceEl = document.querySelector('[class*="ReadableNumber"] span, [class*="readableNumber"]');
                if (priceEl) {
                    const text = priceEl.textContent.trim();
                    const match = text.match(/\$?([0-9,]+\.?[0-9]*)/);
                    if (match) {
                        const price = parseFloat(match[1].replace(/,/g, ''));
                        if (!isNaN(price) && price > 0) return price;
                    }
                }

                // Fallback: generic price selectors
                const rateEls = document.querySelectorAll('[class*="rate"], [class*="price"], [class*="Price"]');
                for (const el of rateEls) {
                    const text = el.textContent.trim();
                    const match = text.match(/\$([0-9,]+\.?[0-9]*)/);
                    if (match) {
                        const price = parseFloat(match[1].replace(/,/g, ''));
                        if (!isNaN(price) && price > 0) return price;
                    }
                }
                return null;
            },
            navigate: (tokenAddress, symbol) => {
                const target = `https://jup.ag/tokens/${tokenAddress}`;
                if (window.location.href !== target) {
                    window.location.href = target;
                }
            }
        },

        BINANCE: {
            name: 'Binance',
            hostMatch: 'binance.com',
            icon: '🔶',
            extract: (url) => {
                const match = url.pathname.match(/\/trade\/([A-Z0-9]+)_([A-Z0-9]+)/i);
                if (match) {
                    return {
                        chain: 'binance',
                        tokenAddress: match[1],
                        symbol: match[1],
                        quoteSymbol: match[2],
                    };
                }
                return null;
            },
            getSymbolFromPage: () => {
                return null;
            },
            getTokenUrl: (symbol) => `https://www.binance.com/en/trade/${symbol}_USDT`,
            getPriceFromPage: () => {
                const match = document.title.match(/^([0-9,]+\.?[0-9]*)/);
                if (match) return parseFloat(match[1].replace(/,/g, ''));
                return null;
            },
            navigate: (tokenAddress, symbol) => {
                const target = `https://www.binance.com/en/trade/${symbol}_USDT`;
                if (window.location.href !== target) {
                    window.location.href = target;
                }
            }
        },

        TELEMETRY: {
            name: 'Telemetry',
            hostMatch: 'telemetry.io',
            icon: '📈',
            extract: (url) => {
                // https://app.telemetry.io/trading/<MINT>
                const match = url.pathname.match(/\/trading\/([a-zA-Z0-9]+)/i);
                if (match) {
                    return {
                        chain: 'solana',
                        tokenAddress: match[1],
                        symbol: null,
                    };
                }
                return null;
            },
            getSymbolFromPage: () => {
                // 1. Title format: "PENGUIN $21.13m - TELEMETRY"
                const titleMatch = document.title.match(/^([A-Za-z0-9$]+)\s/);
                if (titleMatch && titleMatch[1] !== 'TELEMETRY') return titleMatch[1];

                // 2. DOM: span with semibold text in token header area
                const selectors = [
                    'span.text-text-primary.text-base.font-semibold',
                    'span.font-semibold.text-base',
                    'h1', 'h2',
                ];
                for (const sel of selectors) {
                    try {
                        const el = document.querySelector(sel);
                        if (el) {
                            const text = el.textContent.trim();
                            if (text && text.length <= 20 && text !== 'TELEMETRY') return text;
                        }
                    } catch (e) { /* skip */ }
                }

                return null;
            },
            getTokenUrl: (tokenAddress) => `https://app.telemetry.io/trading/${tokenAddress}`,
            getPriceFromPage: () => {
                // 1. Look for price label + sibling value pattern
                const labels = document.querySelectorAll('span, div, p');
                for (const label of labels) {
                    if (label.textContent.trim() === 'Price' && label.children.length === 0) {
                        // Check next sibling or parent's next child
                        const sibling = label.nextElementSibling;
                        if (sibling) {
                            const text = sibling.textContent.trim();
                            const m = text.match(/\$?\s*([0-9,]+\.?[0-9]*)/);
                            if (m) return parseFloat(m[1].replace(/,/g, ''));
                        }
                    }
                }

                // 2. Generic: find elements with $ followed by a number
                const candidates = document.querySelectorAll('span, div');
                for (const el of candidates) {
                    if (el.children.length === 0) {
                        const text = el.textContent.trim();
                        // Match "$0.7444" or "$ 0.7444" but NOT "$21.13m" (market cap)
                        const m = text.match(/^\$\s*([0-9]+\.?[0-9]*)$/);
                        if (m) {
                            const val = parseFloat(m[1]);
                            if (!isNaN(val) && val > 0 && val < 1000000) return val;
                        }
                    }
                }

                // 3. Try broader match for very small prices like $0.0000001234
                for (const el of candidates) {
                    if (el.children.length === 0) {
                        const text = el.textContent.trim();
                        const m = text.match(/^\$\s*(0\.0+[0-9]+)$/);
                        if (m) return parseFloat(m[1]);
                    }
                }

                return null;
            },
            navigate: (tokenAddress, symbol) => {
                const target = `https://app.telemetry.io/trading/${tokenAddress}`;
                if (window.location.href !== target) {
                    window.location.href = target;
                }
            }
        }
    };

    /**
     * Detect current platform from URL
     * @returns {{ platform: Object, tokenInfo: Object } | null}
     */
    function detect(urlString) {
        let url;
        try {
            url = new URL(urlString || window.location.href);
        } catch {
            return null;
        }

        for (const [key, platform] of Object.entries(PLATFORMS)) {
            if (url.hostname.includes(platform.hostMatch)) {
                const tokenInfo = platform.extract(url);
                return {
                    platformKey: key,
                    platformName: platform.name,
                    platformIcon: platform.icon,
                    tokenInfo,
                    getSymbolFromPage: platform.getSymbolFromPage,
                    getTokenMintFromPage: platform.getTokenMintFromPage,
                    getTokenUrl: platform.getTokenUrl,
                    navigate: platform.navigate
                };
            }
        }

        return null;
    }

    /**
     * Try to get symbol from page DOM (platform-specific)
     */
    function getSymbol(detection) {
        if (!detection) return null;
        if (detection.tokenInfo?.symbol) return detection.tokenInfo.symbol;
        if (detection.getSymbolFromPage) {
            return detection.getSymbolFromPage();
        }
        return null;
    }

    /**
     * Navigate to token page on current platform
     */
    function navigate(detection, tokenAddress, symbol) {
        if (detection && detection.navigate) {
            detection.navigate(tokenAddress, symbol);
        } else {
            // Default fallback if no platform detected or no navigate method
            window.location.href = `https://dexscreener.com/solana/${tokenAddress}`;
        }
    }

    return { detect, getSymbol, navigate, PLATFORMS };
})();

// Make available globally for content script
if (typeof window !== 'undefined') {
    window.PlatformDetector = PlatformDetector;
}
